% solve_full_master_equation_main()

% The program calculates the full dynamics of the molecular clutch system
% k_off is different for vinculin-unbound and vinculin-bound talins
% There are no non-talin-based molecular clutches
 

close all

%%%% Model parameters %%%%

nascent_adhesion = true;

all_clutches_initially_attached = false;

a_1 = 0.6;       % [nm], characteristic radius of single talin based clutch adhesion site

v_0 = 120;       % [nm/s], = 110 nm/s [Artola, Nature Materials, 2014], = 110 nm/s [Artola, Nat Cell Biol, 2016], ~200 nm/s [Beausang et al, Biophys. J., 2008, 95:5820]
F_st = 2;        % [pN], = 2 pN [Artola, Nature Materials, 2014], = 2 pN [Artola, Nat Cell Biol, 2016]
sigma_m = 82; %68; %82;  % Myosin density, [1/um^2]. Based on Alieva and Hu papers.
sigma_c = 300;   % [1/um^2], density of sites available for molecular clutch formation (clutches max density = integrin density)

if (nascent_adhesion)
    S_0 = 0.165;       % [um^2] initial area of NAs
else
    S_0 = 9.0;         % [um^2] initial area of FAs
end

k_on = 0.3;                     % [1/s], formation rate of talin-based molecular clutches
k_vinculin_on = 0.03;           % [um^2 / s], characteristic second-order assembly rate of molecular clutches with vinculin (estimated density of talin with vinculins ~ 70% * 300 um^-2 ), the exact value of the parameter is unknown
k_vinculin_talin = 5; % 5; % 1; % [1/s], characteristic first-order binding rate of vinculin to unfolded talin, the mean binding time ~ 45 s
k_off_vinculin_talin = 0.014;   %k_vinculin_talin / 100; [1/s], 0.014 - measured in Yinan, Yan Jie paper for full-length vinculin

Young_modulus = 1e+06;              % Young's modulus of the substrate
k_s_prime = Young_modulus * ( 4 * pi / 9 * a_1 * 10^(-9) * 1000 ); % [pN / nm], spring constant of local substrate deformations

ksi_s = 1;  % [pN * s / nm], viscous coefficient of the substrate

% [Artola 2016] data
k_off_1_talin = 1.6 * 0.3755;         % [1/s], dissociation rate of talin-based molecular clutches, no vinculin (catch bond)
k_off_2_talin = 1.6 * 5.433e-05;      % [1/s], dissociation rate of talin-based molecular clutches, no vinculin (catch bond)
dissociation_force_1_talin = -38.09;  % [pN], characteristic dissociation force of non-talin-based molecular clutches (catch bond)
dissociation_force_2_talin = 4.411;   % [pN], characteristic dissociation force of non-talin-based molecular clutches (catch bond)

k_off_1_talin_vinculin = 0.3755;               % [1/s], dissociation rate of talin-based molecular clutches bound to vinculin (catch bond)
k_off_2_talin_vinculin = 5.433e-05;            % [1/s], dissociation rate of talin-based molecular clutches bound to vinculin (catch bond)
dissociation_force_1_talin_vinculin = -38.09;  % [pN], characteristic dissociation force of non-talin-based molecular clutches (catch bond)
dissociation_force_2_talin_vinculin = 4.411;   % [pN], characteristic dissociation force of non-talin-based molecular clutches (catch bond)


%%% Clutch transition model parameters, single mechano-sensitive domain %%%
 
N_unfolding_domains = 1;      % Here only the mechanosensitive R3 domain of talin is assumed to unfold under mechanical load
 
N_states = 2^N_unfolding_domains;              % domain folded and domain unfolded
list_of_clutch_states = ones(N_states, N_unfolding_domains);
for i = 1 : N_states
    list_of_clutch_states(i, :) = bitget(i, 1:N_unfolding_domains);  % state = 0 - folded, 1 - unfolded
end
 
kT = 1.38e-23 * 310 / 1e-21;  % [pN*nm], thermal energy

k_0_unfold = 0.018;           % [s^-1], zero force unfolding rate of talin [Yao Mingxi article]
k_0_refold = 22.2;            % [s^-1], zero force refolding rate of talin [Yao Mingxi article]
unfolding_force = kT / 5.7;   % [pN], characteristic unfolding force of talin [Yao Mingxi article]

clutch_delta_G_0 = kT * log(k_0_refold ./ k_0_unfold ); % delta_G_0 = G_unfolded - G_folded > 0 in [pN*nm] units
 
if( size(k_0_unfold, 2) ~= N_unfolding_domains || size(k_0_refold, 2) ~= N_unfolding_domains || size(unfolding_force, 2) ~= N_unfolding_domains) 
    disp('Error, wrong length of k_0_unfold, k_0_refold or unfolding_force arrays')
end
 
L_p = 1;                      % [nm] persistence length of a peptide chain, [Winardhi et al, Biophys. J., 2016, 111:2349].
linker_size = 37;             % 37 nm - all domains folded, 217 nm - 3 vinculin binding domains unfolded (VD1 vinculin experiment) the contour length of the flexible talin neck domain in [nm], contour length of talin with folded domains is 97nm [Liu et al, Tony group, PNAS, 2015, E4864]
domain_contour_length = 49.6; % [nm], array (row) of the contour lengths of unfolded clutch domains
 
talin_globular_domain_number = 12;   % 12 - all domains folded, 10 - 3 vinculin binding domains unfolded (VD1 vinculin experiment)
integrin_globular_domain_number = 0; % does not play an important role
globular_domain_size = 5;            % mean size of folded domains in [nm]
 
number_added_sites_per_unfolded_clutch = 2.55;

% talin_K0_unfold = [4.2E-6, 1.7E-8, 0.018, 4.2E-6, 2.5E-5, 2.5E-5, 4.2E-06, 4.2E-6, 2.5E-5, 2.5E-5, 1.7E-8];  % in units of  s^-1, zero force unfolding rates from Table 1
% talin_K0_refold = [0.11, 0.019, 22.2, 0.46, 1.0, 1.0, 0.39, 0.93, 0.93, 0.93, 0.93 ];                        % in units of s^-1, zero force refolding rates from Table 1
% talin_X_unfold  = [3.1, 3.40, 5.7, 3.1, 4.1, 4.1, 3.1, 3.1, 4.1, 4.1, 3.40];                                 % in units of nm, the unfolding transition distance from Table 1
% Rod_contour_length = [69.2, 52.4, 49.6, 52.4, 64, 60.4, 91.6, 66.8, 63.2, 66.4, 62.8]; % [nm]

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


l_step = 0.01; % [nm]
l = 0 : l_step : 1800;

[tabulated_clutch_forces, tabulated_clutch_free_energies] = tabulate_clutch_force_and_energy(k_s_prime, N_unfolding_domains, l, ...
    linker_size, domain_contour_length, clutch_delta_G_0, list_of_clutch_states, kT, L_p, talin_globular_domain_number, ...
    integrin_globular_domain_number, globular_domain_size); 

tabulated_force_derivative = ( circshift(tabulated_clutch_forces, -1, 2) - circshift(tabulated_clutch_forces, 1, 2) ) / 2 / l_step;
tabulated_force_derivative(:, 1) = ( tabulated_clutch_forces(:, 2) - tabulated_clutch_forces(:, 1) ) / l_step;
tabulated_force_derivative(:, end) = ( tabulated_clutch_forces(:, end) - tabulated_clutch_forces(:, end-1) ) / l_step;
        
% force_table = [tabulated_clutch_forces(:, :); tabulated_clutch_forces(1, :)]; % force-response of vinculin-bonund clutches = force-response of unfolded clutches

k_off_talin = k_off_1_talin * exp( tabulated_clutch_forces / dissociation_force_1_talin ) + k_off_2_talin * exp( tabulated_clutch_forces / dissociation_force_2_talin );
k_off_talin( k_off_talin > 1e+2 ) = 1e+2;

k_off_talin_vinculin = k_off_1_talin_vinculin * exp( tabulated_clutch_forces / dissociation_force_1_talin_vinculin ) + k_off_2_talin_vinculin * exp( tabulated_clutch_forces / dissociation_force_2_talin_vinculin );
k_off_talin_vinculin( k_off_talin_vinculin > 1e+2 ) = 1e+2;

k_unfold = k_0_unfold * exp( tabulated_clutch_forces(2, :) / unfolding_force);
k_unfold( k_unfold > 1e+2 ) = 1e+2;
            
k_refold = k_unfold .* exp( (tabulated_clutch_free_energies(1, :) - tabulated_clutch_free_energies(2, :) ) / kT );
k_refold( k_refold > 1e+2 ) = 1e+2;            

N_on = zeros(3, length(l)); % state 1 - unfolded talin, 2 - folded talin, 3 - vinculin-bound talin
if(all_clutches_initially_attached)
    N_on(2, 1) = sigma_c * S_0 / l_step; % all clutches initially are attached and relaxed
end    
N_on_old = zeros(3, length(l));

time_step = 0.00005;
time = 0 : time_step : 181; %300

v_actin = zeros(1, length(time));
F_c = zeros(1, length(time));
x_s = zeros(1, length(time));

N_on_f = zeros(1, length(time));
N_on_u = zeros(1, length(time));
N_on_v = zeros(1, length(time));
N_total = zeros(1, length(time));
N_m = zeros(1, length(time));
N_off = zeros(1, length(time)); 
N_off_talin_vinculin = zeros(1, length(time));
N_total_talin_vinculin = zeros(1, length(time));

F_on_f_total = zeros(1, length(time));
F_on_u_total = zeros(1, length(time));
F_on_v_total = zeros(1, length(time));

F_on_f = zeros(1, length(time));
F_on_u = zeros(1, length(time));
F_on_v = zeros(1, length(time));

Loading_rate_on_f = zeros(1, length(time));
Loading_rate_on_u = zeros(1, length(time));
Loading_rate_on_v = zeros(1, length(time));

cell_adhesion_area = zeros(1, length(time));

cell_adhesion_area(1) = S_0;
N_total(1) = sigma_c * S_0;
N_m(1)     = sigma_m * S_0;

for i = 1 : length(time) - 1

    if(rem(i, 1000) == 0)
        disp(['   i = ' num2str(i)])
    end

    N_on_f(i) = l_step * sum(N_on(2, :), 'all');
    N_on_u(i) = l_step * sum(N_on(1, :), 'all');
    N_on_v(i) = l_step * sum(N_on(3, :), 'all');

    N_off(i) = N_total(i) - N_on_f(i) - N_on_u(i) - N_on_v(i);
    N_off_talin_vinculin(i) = N_total_talin_vinculin(i) - N_on_v(i);
    
    if(N_off(i) < 0) 
        N_off(i) = 0; 
    end

    if(N_off_talin_vinculin(i) < 0)
        N_off_talin_vinculin(i) = 0;
    end

    F_on_f_total(i) = sum( tabulated_clutch_forces(2, :) .* N_on(2, :), 'all' ) * l_step;
    F_on_u_total(i) = sum( tabulated_clutch_forces(1, :) .* N_on(1, :), 'all' ) * l_step;
    F_on_v_total(i) = sum( tabulated_clutch_forces(1, :) .* N_on(3, :), 'all' ) * l_step;

    F_on_f(i) = F_on_f_total(i) / N_on_f(i);
    F_on_u(i) = F_on_u_total(i) / N_on_u(i);
    F_on_v(i) = F_on_v_total(i) / N_on_v(i);
   
    Loading_rate_on_f(i) = v_actin(i) * sum( tabulated_force_derivative(2, :) .* N_on(2, :), 'all' ) * l_step / N_on_f(i);
    Loading_rate_on_u(i) = v_actin(i) * sum( tabulated_force_derivative(1, :) .* N_on(1, :), 'all' ) * l_step / N_on_u(i);
    Loading_rate_on_v(i) = v_actin(i) * sum( tabulated_force_derivative(1, :) .* N_on(3, :), 'all' ) * l_step / N_on_v(i);

    N_on_old = N_on;    % vector
    
    F_c(i + 1) = F_on_f_total(i) + F_on_u_total(i) + F_on_v_total(i);
    if (nascent_adhesion)
        cell_adhesion_area(i + 1) = N_total(i) / sigma_c;    
    else
        cell_adhesion_area(i + 1) = S_0;   % myosin number
    end
    N_m(i + 1) = cell_adhesion_area(i + 1) * sigma_m;   % myosin number
    v_actin(i + 1) = v_0 * ( 1 - F_c(i + 1) / N_m(i + 1) / F_st );
    
    a_2 = 1000 * (cell_adhesion_area(i + 1) / pi)^0.5; % [nm]
    k_s = Young_modulus * ( 4 * pi / 9 * a_2 * 10^(-9) * 1000 ); % [pN / nm]
    if(ksi_s < 0.001)
        x_s(i + 1) = F_c(i + 1) / k_s;
        d_x_s_d_t = (x_s(i + 1) - x_s(i)) / time_step;
    else
        d_x_s_d_t = (F_c(i + 1) - k_s * x_s(i)) / ksi_s;
    end
    x_s(i + 1) = x_s(i) + time_step * d_x_s_d_t;

    d_N_on_d_l = ( N_on - circshift(N_on, [0 1]) ) / l_step;
    d_N_on_d_l(:, 1) = N_on(:, 1) / l_step;

%     d_sigma_on_d_l = ( circshift(sigma_on, [0 -1]) - circshift(sigma_on, [0 1]) ) / 2 / l_step;
%     d_sigma_on_d_l(:, end) = ( sigma_on(:, end) - sigma_on(:, end-1) ) / l_step;

    N_on = N_on_old - time_step * (v_actin(i + 1) - d_x_s_d_t) * d_N_on_d_l;
    
    N_on(1, :) = N_on(1, :) - time_step * k_off_talin(1, :) .* N_on_old(1, :)          + time_step * k_unfold .* N_on_old(2, :)        - time_step * k_refold .* N_on_old(1, :) - time_step * k_vinculin_talin * N_on_old(1, :) + time_step * k_off_vinculin_talin * N_on_old(3, :);
    N_on(2, :) = N_on(2, :) - time_step * k_off_talin(2, :) .* N_on_old(2, :)          - time_step * k_unfold .* N_on_old(2, :)        + time_step * k_refold .* N_on_old(1, :);
    N_on(3, :) = N_on(3, :) - time_step * k_off_talin_vinculin(1, :) .* N_on_old(3, :) + time_step * k_vinculin_talin * N_on_old(1, :) - time_step * k_off_vinculin_talin * N_on_old(3, :);
        
    N_on(2, 1) = N_on(2, 1) + time_step * k_on * N_off(i) / l_step;
    N_on(3, 1) = N_on(3, 1) + time_step * k_vinculin_on * N_off(i) * N_off_talin_vinculin(i) / cell_adhesion_area(i + 1) / l_step;

    N_on( N_on < 0 ) = 0;

    N_total_talin_vinculin(i + 1) = N_total_talin_vinculin(i) + time_step * k_vinculin_talin * N_on_u(i) - time_step * k_off_vinculin_talin * N_total_talin_vinculin(i);

    N_total(i + 1) = N_total(1) + number_added_sites_per_unfolded_clutch * N_total_talin_vinculin(i);
%     sigma_total(i + 1) = sigma_total(i) + (sigma_total_max - sigma_total(i)) * k_talin_sites_accumulation * time_step;
    
end

%%% Stationary p_on and F_c calculation %%%

% k_c = 0.05; % [pN/nm], the linear WT model 
% l_y = 1000; % [nm], the linear WT model 
% N_c = N_total(end);   % clutch number
% k = k_s_prime * k_c / (k_s_prime + k_c);
% 
% % k_off_fun = @(x) k_off_with_yield_point(x, l_y, k, k_off_1, k_off_2, dissociation_force_1, dissociation_force_2);
% force_fun = @(x) force_with_yield_point(x, l_y, k);
% 
% clutches_k_off_integral = @(x) k_off_integral_catch_bond_v4c(x, l_y, k, k_off_1, k_off_2, dissociation_force_1, dissociation_force_2);
% v = v_actin(end);
% 
% warning('off', 'MATLAB:integral:NonFiniteValue')
%             
% integrand_fun_1 = @(x) exp(-clutches_k_off_integral(x) / v);
% C_coeff = k_on / (k_on * integral( integrand_fun_1, 0, Inf ) + v);
% p_on_stationary = @(x) C_coeff * exp(-clutches_k_off_integral(x) / v);
% 
% integrand_fun_2 = @(x) force_fun(x) .* p_on_stationary(x);
% force_stationary = N_c * integral( integrand_fun_2, 0, Inf );
%             
% warning('on','all')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

figure(1)
plot(time, v_actin)
hold on
title('Actin flow')
xlabel('Time, s') 
ylabel('Actin flow, nm/s') 

figure(2)
plot(time, F_c ./ cell_adhesion_area)
hold on
title('Cell traction')
xlabel('Time, s') 
ylabel('Cell traction, Pa')

figure(3)
semilogy(l, N_on(2, :), 'k') % folded clutches
hold on
semilogy(l, N_on(1, :), 'r') % unfolded clutches
semilogy(l, N_on(3, :), 'b') % vinculin-bound clutches
% semilogy(l, p_on_stationary(l), 'g--')

ylim([0.00001 10])

title('Clutch number density')
xlabel('Extension, nm') 
ylabel('Clutch number density') 

figure(4)
plot(time, x_s, 'k')
hold on
title('Substrate stretching')
xlabel('Time, s') 
ylabel('Substrate coordinate, nm')

figure(5)
semilogy(time, N_on_f, 'k') % folded clutches
hold on
semilogy(time, N_on_u, 'r') % unfolded clutches
semilogy(time, N_on_v, 'b') % vinculin-bound clutches
semilogy(time, N_total_talin_vinculin, 'b--') % total vinculin-bound talins
semilogy(time, 2*N_total_talin_vinculin, 'm--') % total vinculin in cell adhesion (assuming that R3 domain has 2 binding sites for vinculin)

% ylim([0.1 2000])
ylim([0.001 100])

title('Clutch number')
xlabel('Time, s') 
ylabel('Clutch number')

figure(6)
semilogy(time, F_on_f, 'k') % folded clutches
hold on
semilogy(time, F_on_u, 'r') % unfolded clutches
semilogy(time, F_on_v, 'b') % vinculin-bound clutches

ylim([0.1 100])

title('Clutch tension')
xlabel('Time, s') 
ylabel('Clutch tension, pN')

figure(7)
semilogy(time, Loading_rate_on_f, 'k') % folded clutches
hold on
semilogy(time, Loading_rate_on_u, 'r') % unfolded clutches
semilogy(time, Loading_rate_on_v, 'b') % vinculin-bound clutches

title('Clutch loading rate')
xlabel('Time, s') 
ylabel('Clutch loading rate, pN/s')

ylim([0.1 100])

figure(8)
plot(time, N_total_talin_vinculin, 'b') % total vinculin-bound talins
hold on
plot(time, 2*N_total_talin_vinculin, 'm--') % total vinculin in cell adhesion (assuming that R3 domain has 2 binding sites for vinculin)

title('Vinculin number')
xlabel('Time, s') 
ylabel('Vinculin number')

figure(9)
plot(time, cell_adhesion_area, 'b') % total vinculin-bound talins
hold on
title('FA area')
xlabel('Time, s') 
ylabel('FA area, um^2')

%%% Output results %%%

t = time(1 : 1000 : end);
S = cell_adhesion_area( 1 : 1000 : end);
v = v_actin(1 : 1000 : end);
P = F_c( 1 : 1000 : end) ./ S;
vinculin_total = 2*N_total_talin_vinculin( 1 : 1000 : end);
talin_total    = N_total_talin_vinculin( 1 : 1000 : end) + N_on_f( 1 : 1000 : end) + N_on_u( 1 : 1000 : end);

loading_rate_f = mean(Loading_rate_on_f(end-100:end))
loading_rate_u = mean(Loading_rate_on_u(end-100:end))
loading_rate_v = mean(Loading_rate_on_v(end-100:end))

N_f = mean(N_on_f(end-100:end))
N_u = mean(N_on_u(end-100:end))
N_v = mean(N_on_v(end-100:end))
N_c = mean(N_total(end-100:end))

loading_rate_mean = (N_f * loading_rate_f + N_u * loading_rate_u) / (N_f + N_u)
loading_rate_mean = (N_f * loading_rate_f + N_u * loading_rate_u + N_v * loading_rate_v) / (N_f + N_u + N_v)

N_f = N_on_f( 1 : 1000 : end);
N_u = N_on_u( 1 : 1000 : end);
N_v = N_on_v( 1 : 1000 : end);
t = time(1 : 1000 : end);
S = cell_adhesion_area( 1 : 1000 : end);
N_c = N_total( 1 : 1000 : end);